<?php
/**
 * Plugin Name: DPD France
 * Plugin URI: http://www.dpd.fr/ecommerce
 * Description: Module de livraison DPD pour WooCommerce 2 & 3 (France uniquement)
 * Version: 5.3.3
 * Author: DPD France S.A.S.
 * Author URI: http://www.dpd.fr/
 * License: Open Software License (OSL 3.0) - http://opensource.org/licenses/osl-3.0.php
 * Text Domain: woocommerce-dpdfrance
 */

/* Exit if accessed directly */
if ( !defined( 'ABSPATH' ) ) {
	exit;
}

define( 'OS_DPDFRANCE_FILE_PATH', plugin_dir_path( __FILE__ ) );
define( 'OS_DPDFRANCE_ROOT_URL', plugins_url( '', __FILE__ ) );

class OS_dpdfrance_shipping_wrapper {
	/**
	 * OS_dpdfrance_shipping_wrapper constructor.
	 */
	public function __construct() {
	}

	public function OS_dpdfrance_shipping_main() {
		include_once( ABSPATH . 'wp-admin/includes/plugin.php' );
		if ( is_plugin_active( 'woocommerce/woocommerce.php' ) ) {
			add_action( 'plugins_loaded', [ $this, 'init' ], 8 );
		}
	}

    /* * Equivalent du construct */
	public function init() {
		require_once( OS_DPDFRANCE_FILE_PATH . '/settings/settings-dpdfrance-classic.php' );
		require_once( OS_DPDFRANCE_FILE_PATH . '/settings/settings-dpdfrance-predict.php' );
		require_once( OS_DPDFRANCE_FILE_PATH . '/settings/settings-dpdfrance-relais.php' );
		require_once( OS_DPDFRANCE_FILE_PATH . '/settings/settings-dpdfrance-world.php' );
		require_once( OS_DPDFRANCE_FILE_PATH . '/settings/settings-dpdfrance-admin.php' );
		require_once( OS_DPDFRANCE_FILE_PATH . '/classes/class-wc-dpdfrance-constants.php' );
		require_once( OS_DPDFRANCE_FILE_PATH . '/classes/class-wc-dpdfrance-tools.php' );

		/* * Install DPD Relais */
		require_once( OS_DPDFRANCE_FILE_PATH . '/classes/class-wc-dpdfrance-relais.php' );
		woocommerce_dpdfrance_relais_init();

		/* * Install DPD Predict */
		require_once( OS_DPDFRANCE_FILE_PATH . '/classes/class-wc-dpdfrance-predict.php' );
		woocommerce_dpdfrance_predict_init();

		/* * Install DPD Classic */
		require_once( OS_DPDFRANCE_FILE_PATH . '/classes/class-wc-dpdfrance-classic.php' );
		woocommerce_dpdfrance_classic_init();

		/* * Install DPD Europe & Intercontinental */
		require_once( OS_DPDFRANCE_FILE_PATH . '/classes/class-wc-dpdfrance-world.php' );
		woocommerce_dpdfrance_world_init();

		/* * Add DPD France Tools and Settings tabs */
		add_action( 'admin_menu', [ $this, 'add_export_tab' ] );
		add_action( 'admin_init', [ $this, 'export_dpdfrance_orders' ] );
		add_action( 'woocommerce_display_admin_footer_text', [ $this, 'dpdfrance_syncshipments' ] );
		add_filter( 'woocommerce_settings_tabs_array', [ $this, 'add_settings_tab' ], 50 );
		add_action( 'woocommerce_settings_tabs_dpdfrance_shipping', [ $this, 'settings_tab' ] );
		add_action( 'woocommerce_update_options_dpdfrance_shipping', [ $this, 'update_settings' ] );
		add_action( 'admin_print_styles', [$this, 'woocommerce_dpdfrance_admin_style'] );

		/* * Load translations */
		load_plugin_textdomain( 'woocommerce-dpdfrance', false, dirname( plugin_basename( __FILE__ ) ) . '/languages/' );
	}

	/**
     * * Uninstall plugin and DB tables
     */
	public function deactivate() {
		include_once( 'controllers/admin/dpdfrance-shipping-install-table.php' );
		uninstall_table();
	}

	/**
     * * Installs plugin
     */
	public function install() {
		global $wp_version;

		if ( !is_plugin_active( 'woocommerce/woocommerce.php' ) ) {
			deactivate_plugins( plugin_basename( __FILE__ ) ); /* Deactivate plugin */
			wp_die(
				__( 'You must run at least WooCommerce version 2.x to install DPD delivery plugin', 'woocommerce-dpdfrance' ),
				__( 'WC not activated', 'woocommerce-dpdfrance' ),
				[ 'back_link' => true ]
			);
            return;
		}

		if ( (float) $wp_version < 3.5 ) {
			deactivate_plugins( plugin_basename( __FILE__ ) ); /* Deactivate plugin */
			wp_die(
				__( 'You must run at least WordPress version 3.5 to install DPD delivery plugin', 'woocommerce-dpdfrance' ),
				__( 'WP not compatible', 'woocommerce-dpdfrance' ),
				[ 'back_link' => true ]
			);
            return;
		}

		if ( !extension_loaded('xml') ) {
			deactivate_plugins( plugin_basename( __FILE__ ) ); /* Deactivate plugin */
			wp_die(
				__( 'The PHP extension XML is not installed on this server. You must activate it in order to use the DPD plugin', 'woocommerce-dpdfrance' ),
				__( 'SOAP not installed', 'woocommerce-dpdfrance' ),
				[ 'back_link' => true ]
			);
            return;
		}

        if ( !extension_loaded( 'soap' ) ) {
            deactivate_plugins( plugin_basename( __FILE__ ) ); /* Deactivate plugin */
            wp_die(
                __( 'The PHP extension SOAP is not installed on this server. You must activate it in order to use the DPD plugin', 'woocommerce-dpdfrance' ),
                __( 'SOAP not installed', 'woocommerce-dpdfrance' ),
                [ 'back_link' => true ]
            );
            return;
        }

		define( 'OS_DPDFRANCE_FILE_PATH', __DIR__ );

		/* Install DB tables */
		include_once( 'controllers/admin/dpdfrance-shipping-install-table.php' );
		install_table();
	}

	/**
	 * [BO][WooCommerce][DPD France] Add DPD France under WooCommerce section in the sidebar
	 */
	public function add_export_tab() {
		add_submenu_page(
			'woocommerce',
			__( 'DPD France', 'woocommerce-dpdfrance' ),
			__( 'DPD France', 'woocommerce-dpdfrance' ),
			'manage_woocommerce',
			'woocommerce-dpdfrance',
			[ $this, 'display_export_page' ],
			8
		);
	}

	/**
	 * [BO][WooCommerce][DPD France] Orders management page
	 */
	public function display_export_page() {
		global $wpdb;

		//* Display build
		//* Loads scripts and page header
		?>
        <link rel="stylesheet" type="text/css" href="<?php echo OS_DPDFRANCE_ROOT_URL; ?>/assets/js/back/jquery/plugins/fancybox/jquery.fancybox.min.css"/>
        <script type="text/javascript" src="<?php echo OS_DPDFRANCE_ROOT_URL; ?>/assets/js/back/jquery/plugins/marquee/jquery.marquee.min.js"></script>
        <script type="text/javascript" src="<?php echo OS_DPDFRANCE_ROOT_URL; ?>/assets/js/back/jquery/plugins/fancybox/jquery.fancybox.min.js"></script>
        <script type="text/javascript" src="<?php echo OS_DPDFRANCE_ROOT_URL; ?>/assets/js/back/main.js"></script>

        <div class="dpdfrance-wrap">
            <h2>
                <img src="<?php echo OS_DPDFRANCE_ROOT_URL; ?>/assets/img/admin/admin.png"/> Gestion des expéditions
            </h2>
		<?php

		// * RSS stream
		$this->display_dpd_rss_flux();

		// * Filter field
		echo '<input id="tableFilter" placeholder="' . __( 'Search something, separate values by semicolons ;', 'woocommerce-dpdfrance' ) . '"/>
		      <img id="filtericon" src="' . OS_DPDFRANCE_ROOT_URL . '/assets/img/admin/search.png"/><br/><br/>';

		// * POST action : updateShippedOrders | On click change selected orders to currently shipped | wc-processing
		if ( isset( $_POST['updateShippedOrders'] ) ) {
			if ( !isset( $_POST['checkbox'] ) ) {
				echo '<div class="warnmsg">' . __( 'No order selected', 'woocommerce-dpdfrance' ) . '</div>';
			} else {
				foreach ( $_POST['checkbox'] as $order_id ) {
					$order        = wc_get_order( $order_id );
					$depot_code   = WC_DPDFrance_Tools::get_depot_code( $order );
					$shipper_code = WC_DPDFrance_Tools::get_shipper_code( $order );

					// * Prepare customer note
					$note = __( 'Dear customer, you can follow your DPD parcel delivery by clicking this link:', 'woocommerce-dpdfrance' );
					$link = 'http://www.dpd.fr/tracex_' . $order_id . '_' . $depot_code . $shipper_code;
					$href = '<a href="' . $link . '">' . $link . '</a>';

					// * Add customer note to order
					if ( is_user_logged_in() && current_user_can( 'manage_woocommerce' ) ) {
						$user                 = get_user_by( 'id', get_current_user_id() );
						$comment_author       = $user->display_name;
						$comment_author_email = $user->user_email;
					} else {
						$comment_author       = __( 'WooCommerce', 'woocommerce-dpdfrance' );
						$comment_author_email = strtolower( __( 'WooCommerce', 'woocommerce-dpdfrance' ) ) . '@';
						$comment_author_email .= isset( $_SERVER['HTTP_HOST'] ) ? str_replace( 'www.', '', $_SERVER['HTTP_HOST'] ) : 'noreply.com';
						$comment_author_email = sanitize_email( $comment_author_email );
					}

					$comment_post_ID    = $order_id;
					$comment_author_url = '';
					$comment_content    = $note . ' ' . $href;
					$comment_agent      = 'WooCommerce';
					$comment_type       = 'order_note';
					$comment_parent     = 0;
					$comment_approved   = 1;
					$commentdata        = apply_filters( 'woocommerce_new_order_note_data', compact(
						'comment_post_ID',
						'comment_author',
						'comment_author_email',
						'comment_author_url',
						'comment_content',
						'comment_agent',
						'comment_type',
						'comment_parent',
						'comment_approved'
					), [
						'order_id'         => $order_id,
						'is_customer_note' => 1
					] );

					$comment_id = wp_insert_comment( $commentdata );
					add_comment_meta( $comment_id, 'is_customer_note', 1 );
					do_action( 'woocommerce_new_customer_note', [
						'order_id'      => $order_id,
						'customer_note' => $note . ' ' . $href
					] );

					/* Update order status */
					$order->update_status( get_option( 'wc_settings_tab_dpdfrance_etape_expediee' ) );
				}

				/* Display confirmation message */
				echo '<div class="okmsg">' . __( 'Shipped orders statuses were updated', 'woocommerce-dpdfrance' ) . '</div>';
			}
		}

		// * POST action : updateDeliveredOrders | On click change selected orders to delivered | wc-completed
		if ( isset( $_POST['updateDeliveredOrders'] ) ) {
			if ( !isset( $_POST['checkbox'] ) ) {
				echo '<div class="warnmsg">' . __( 'No order selected', 'woocommerce-dpdfrance' ) . '</div>';
			} else {
				foreach ( $_POST['checkbox'] as $order_id ) {
					$order = new WC_Order( $order_id );
					/* Update order status */
					$order->update_status( get_option( 'wc_settings_tab_dpdfrance_etape_livre' ) );
				}

				/* Display confirmation message */
				echo '<div class="okmsg">' . __( 'Delivered orders statuses were updated', 'woocommerce-dpdfrance' ) . '</div>';
			}
		}

		/* Init vars */
		$order_data = [];
		$post_ids   = WC_DPDFrance_Tools::get_orders_id_except_delivered_and_completed_status();

		/* Table header */
		echo '
        <form id="exportform" action="admin.php?page=woocommerce-dpdfrance" method="POST" enctype="multipart/form-data">
        <table class="wp-list-table widefat fixed posts">
            <thead>
                <tr>
                    <th scope="col" id="checkbox" class="manage-column column-cb check-column" style=""><label class="screen-reader-text" for="cb-select-all-1">' . __( '', 'woocommerce-dpdfrance' ) . '</label><input onchange="checkallboxes(this)" id="cb-select-all-1" type="checkbox"/></th>
                    <th scope="col" id="order_id" class="manage-column column-order_id" style="">' . __( 'Order ID', 'woocommerce-dpdfrance' ) . '</th>
                    <th scope="col" id="order_date" class="manage-column column-order_date" style="">' . __( 'Date', 'woocommerce-dpdfrance' ) . '</th>
                    <th scope="col" id="order_customer" class="manage-column column-order_customer" style="">' . __( 'Customer', 'woocommerce-dpdfrance' ) . '</th>
                    <th scope="col" id="order_shipping_method" class="manage-column column-order_shipping_method"  style="">' . __( 'Service', 'woocommerce-dpdfrance' ) . '</th>
                    <th scope="col" id="order_address" class="manage-column column-order_address" style="">' . __( 'Destination', 'woocommerce-dpdfrance' ) . '</th>
                    <th scope="col" id="order_weight" class="manage-column column-order_weight" style="">' . __( 'Weight', 'woocommerce-dpdfrance' ) . '</th>
                    <th scope="col" id="order_amount" class="manage-column column-order_amount" colspan="2" style="">' . __( 'Amount (tick to insure this parcel)', 'woocommerce-dpdfrance' ) . '</th>
                    ' . ( get_option( 'wc_settings_tab_dpdfrance_retour_option' ) == 0 ? '' : '<th scope="col" id="order_retour" class="manage-column column-order_retour" style="">' . __( 'Allow returns', 'woocommerce-dpdfrance' ) . '</th>' ) . '
                    <th scope="col" id="order_status" class="manage-column column-order_status" style="">' . __( 'Order Status', 'woocommerce-dpdfrance' ) . '</th>
                    <th scope="col" id="order_tracking" class="manage-column column-order_tracking" style="">' . __( 'Parcel trace', 'woocommerce-dpdfrance' ) . '</th>
                </tr>
            </thead>
            <tbody id="the-list">';

		/* Collect order data */
		foreach ( $post_ids as $post_id ) {
			/* Retrieve order details from its ID */
			$order                    = wc_get_order( $post_id );
			$order_id                 = $order->get_order_number();
			$order_shipping_method_id = WC_DPDFrance_Tools::get_order_shipping_method_id( $order );

			// * Filter orders not carrier by DPD
			if ( stripos( $order_shipping_method_id, 'dpdfrance_' ) !== false ) {
				// * Assembles order data in array
				$order_data[ $order_id ] = $this->format_order_data( $order, $order_shipping_method_id );
				$address                 = WC_DPDFrance_Tools::get_address_link( $order_data[ $order_id ] );
				$icon                    = WC_DPDFrance_Tools::get_shipping_method_icon( $order_data[ $order_id ] );
				$depot_code              = WC_DPDFrance_Tools::get_depot_code( $order_data[ $order_id ] );
				$shipper_code            = WC_DPDFrance_Tools::get_shipper_code( $order_data[ $order_id ] );

				$tracking                = ( $order_data[ $order_id ]['order_status'] === substr( get_option( 'wc_settings_tab_dpdfrance_etape_expediee' ), 3 ) ?
					'<a target="_blank" href="http://www.dpd.fr/tracex_' . $order_data[ $order_id ]['order_id'] . '_' . $depot_code . $shipper_code . '">
                        <img src="' . OS_DPDFRANCE_ROOT_URL . '/assets/img/admin/tracking.png" alt="Trace"/>
                    </a>' :
					''
				);

				/* Add table row */
				echo '  <tr>
                            <td><input class="checkbox" type="checkbox" name="checkbox[]" value="' . $order_data[ $order_id ]['order_id'] . '" ' . ( strpos( substr( get_option( 'wc_settings_tab_dpdfrance_etape_expedition' ), 3 ), $order_data[ $order_id ]['order_status'] ) !== false ? 'checked=checked' : '' ) . '></td>
                            <td class="id">' . $order_data[ $order_id ]['order_id'] . '</td>
                            <td class="date">' . $order_data[ $order_id ]['order_date'] . '</td>
                            <td class="nom">' . $order_data[ $order_id ]['shipping_first_name'] . ' ' . $order_data[ $order_id ]['shipping_last_name'] . '</td>
                            <td class="type">' . $icon . '</td>
                            <td class="pr">' . $address . '</td>
                            <td class="poids"><input name="poids[' . $order_data[ $order_id ]['order_id'] . ']" class="poids" value="' . number_format( $order_data[ $order_id ]['order_weight'], ( get_option( 'woocommerce_weight_unit' ) == 'g' ? 0 : 2 ), '.', '' ) . '"></input> ' . get_option( 'woocommerce_weight_unit' ) . '</td>
                            <td class="prix" align="right">' . number_format( $order_data[ $order_id ]['order_amount'], 2, '.', '' ) . ' €</td>
                            <td class="advalorem"><input class="advalorem" type="checkbox" name="advalorem[' . $order_data[ $order_id ]['order_id'] . ']" value="' . $order_data[ $order_id ]['order_amount'] . '" ' . ( get_option( 'wc_settings_tab_dpdfrance_advalorem_option' ) == 1 ? 'checked=checked' : '' ) . '></td>
                            ' . ( get_option( 'wc_settings_tab_dpdfrance_retour_option' ) == 0 ? '' : '<td class="retour"><input class="retour" type="checkbox" name="retour[' . $order_data[ $order_id ]['order_id'] . ']" value="' . $order_data[ $order_id ]['order_id'] . '" ' . ( get_option( 'wc_settings_tab_dpdfrance_retour_option' ) == 0 ? '' : 'checked=checked' ) . '></td>' ) . '
                            <td class="statutcommande" align="center">' . wc_get_order_status_name( $order_data[ $order_id ]['order_status'] ) . '</td>
                            <td class="statutcolis" align="center"><a href="javascript:void(0)" onclick="window.open(\'http://www.dpd.fr/tracex_' . $order_data[ $order_id ]['order_id'] . '_' . $depot_code . $shipper_code . '\',\'\',\'width=1024,height=768,top=30,left=20\')"></a>' . $tracking . '</td>
                       </tr>
                ';
			}
		}

		// * If there are no DPD orders, quit
		if ( empty( $order_data ) ) {
			wp_die( '<div class="warnmsg">' . __( 'There are no DPD orders', 'woocommerce-dpdfrance' ) . '</div>' );
			exit;
		}

		// * Display end of table and footer
		echo '        </tbody>
                </table>
                <p>
                    <input type="submit" class="button" name="exportOrders" value="' . __( 'Export selected orders', 'woocommerce-dpdfrance' ) . '"/>
                    <input type="submit" class="button" name="updateShippedOrders" value="' . __( 'Update shipped orders', 'woocommerce-dpdfrance' ) . '"/>
                    <input type="submit" class="button" name="updateDeliveredOrders" value="' . __( 'Update delivered orders', 'woocommerce-dpdfrance' ) . '"/>
                </p>
            </form>
        ';
	}

	/**
	 * [BO][WooCommerce][DPD France] Exporte DAT File
	 */
	public function export_dpdfrance_orders() {
		require_once( OS_DPDFRANCE_FILE_PATH . '/classes/class-wc-dpdfrance-station.php' );
		$order_data = null;

		if ( isset( $_POST['exportOrders'] ) ) {
			/* Init vars */
			$post_ids = WC_DPDFrance_Tools::get_orders_id_except_delivered_and_completed_status();

			/* If there are no DPD orders, quit */
			if ( empty( $post_ids ) ) {
				wp_die( '<div class="warnmsg">' . __( 'There are no DPD orders', 'woocommerce-dpdfrance' ) . '</div>' );
				exit;
			}

			// * Collect order data
			foreach ( $post_ids as $post_id ) {

				// * Retrieve order details from its ID
				$order                    = wc_get_order( $post_id );
				$order_id                 = $order->get_order_number();
				$order_shipping_method_id = WC_DPDFrance_Tools::get_order_shipping_method_id( $order );

				if ( stripos( $order_shipping_method_id, 'dpdfrance_' ) !== false ) {
					// * Assembles order data in array
					$order_data[ $order_id ] = $this->format_order_data( $order, $order_shipping_method_id );
				}
			}

			$order_list = $_POST['checkbox'] ?? null;

			if ( empty( $order_list ) ) {
				echo '<div class="warnmsg">' . __( 'No order selected', 'woocommerce-dpdfrance' ) . '</div>';
			} else {
				// * Init DPDStation class for interface file creation
				$dpdstation = new DPDStation();
				foreach ( $order_list as $order_id ) {
					if ( array_key_exists( $order_id, $order_data ) ) {
						// * Format row of DAT file for DPD France interface
						$dpdstation->formatRow( $order_data[ $order_id ] );
					}
				}
				// * Download file
				$dpdstation->download();
			}
		}
	}

	/**
	 * [BO] Add Auto update in admin footer if option is enabled
	 */
	public function dpdfrance_syncshipments() {
		$cron_url = OS_DPDFRANCE_ROOT_URL . '/ajax/syncshipments.php';
		if ( get_option( 'wc_settings_tab_dpdfrance_auto_update' ) === "1" ) {
			echo '<script type="text/javascript">';
			echo 'jQuery.get("' . $cron_url . '");';
			echo '</script>';
		}
	}

	/**
	 * [BO][WooCommerce][DPD France] Add DPD France settings tab
	 *
	 * @param $settings_tabs
	 *
	 * @return mixed
	 */
	public function add_settings_tab( $settings_tabs ) {
		$settings_tabs['dpdfrance_shipping'] = __( 'DPD France', 'woocommerce-dpdfrance' );

		return $settings_tabs;
	}

	/**
	 *  * [BO][WooCommerce][Reglages][DPD France][Header] Builds DPD France settings tab display
	 */
	public function settings_tab() {
		echo "
            <style media='screen'></style>
            <div class='updated woocommerce-message'>
                <p><strong>" . __( 'Welcome to DPD France.', 'woocommerce-dpdfrance' ) . "</strong> " . __( 'The plugin documentation is available here :', 'woocommerce-dpdfrance' ) . "<a target=\"_blank\" href=\"" . OS_DPDFRANCE_ROOT_URL . "/assets/docs/readme_dpdfrance_woocommerce.pdf\"><img src=\"" . OS_DPDFRANCE_ROOT_URL . "/assets/img/admin/pdf.png\"/></a>
                <p><strong>Important : </strong>" . __( 'You must be a DPD France customer and be in posession of your contract details before configuring this module. If you want to get in touch with our sales team, please visit our website', 'woocommerce-dpdfrance' ) . " <a target='_blank' href='http://www.dpd.fr/professionnels'>www.dpd.fr</a>.</p>
                <p><strong>" . __( 'Please proceed to the configuration of the module', 'woocommerce-dpdfrance' ) . "</strong></p>
            </div>";
		echo "<h3>" . __( 'Your shipper data', 'woocommerce-dpdfrance' ) . "</h3>";
		woocommerce_admin_fields( WC_DPDFrance_Settings_Admin::settings_admin_shipperdata() );
		echo "<h3>" . __( 'Orders management', 'woocommerce-dpdfrance' ) . "</h3>";
		woocommerce_admin_fields( WC_DPDFrance_Settings_Admin::settings_admin_shipments() );
		echo "<h3>" . __( 'Other settings', 'woocommerce-dpdfrance' ) . "</h3>";
		woocommerce_admin_fields( WC_DPDFrance_Settings_Admin::settings_admin_googleapi() );
	}

	/**
     * * [BO][WooCommerce][Reglages][DPD France] Update all settings
     */
	public function update_settings() {
		woocommerce_update_options( WC_DPDFrance_Settings_Admin::settings_admin_shipperdata() );
		woocommerce_update_options( WC_DPDFrance_Settings_Admin::settings_admin_shipments() );
		woocommerce_update_options( WC_DPDFrance_Settings_Admin::settings_admin_googleapi() );
	}

	/**
	 * Display DPD France rss flux
	 */
	public function display_dpd_rss_flux() {
		$dpd_flux_infos = 'http://www.dpd.fr/extensions/rss/flux_info_dpdfr.xml';
		$rss            = @simplexml_load_string( file_get_contents( $dpd_flux_infos ) );
		if ( !empty( $rss->channel->item ) ) {
			$stream = '';
			foreach ( $rss->channel->item as $item ) {
				$stream .= '<strong>' . $item->category . ' > ' . $item->title . ' : </strong> ' . $item->description . '&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;';
			}
			if ( $stream ) {
				$html = '<fieldset>';
				$html .= '      <legend>';
				$html .= '            <a href="javascript:void(0)" onclick="$(\'#zonemarquee\').toggle(\'fast\', function() {var text = $(\'#showhide\').text();$(\'#showhide\').text(text === \'+\' ? \'-\' : \'+\')})">';
				$html .= '                  <img src="' . OS_DPDFRANCE_ROOT_URL . '/assets/img/admin/rss_icon.png" />';
				$html .= __( 'DPD News', 'woocommerce-dpdfrance' );
				$html .= '                  <div id="showhide">-</div>';
				$html .= '            </a>';
				$html .= '      </legend>';
				$html .= '      <div id="zonemarquee">';
				$html .= '            <div id="marquee" class="marquee">' . $stream . '</div>';
				$html .= '      </div>';
				$html .= '</fieldset><br/>';
				echo $html;
			}
		}
	}

	/**
	 * Return formatted order data array
	 *
	 * @param $order
	 * @param $order_shipping_method_id
	 *
	 * @return array
	 */
	public function format_order_data( $order, $order_shipping_method_id ): array {
		$date_created = null;

		// * WC 3+ : Retrieves correct date and time from post creation timestamp
		if ( WC_DPDFrance_Tools::is_from_wc3() ) {
			$post_timestamp = new DateTime();
			$post_timestamp->setTimestamp( strtotime( $order->get_date_created() ) );
			$date_created = $post_timestamp->setTimezone( new DateTimeZone( 'Europe/Paris' ) )->format( 'd/m/Y H:i:s' );
		}
		$order_weight  = WC_DPDFrance_Tools::get_order_total_weight( $order );
		$customer_note = str_replace(
			[ "\r\n", "\n", "\r", "\t" ],
			' ',
			( WC_DPDFrance_Tools::is_from_wc3() ? $order->get_customer_note() : $order->customer_note )
		);
		$order_date    = ( WC_DPDFrance_Tools::is_from_wc3() ? $date_created : date( 'd/m/Y H:i:s', strtotime( $order->order_date ) ) );

		return [
			'order_id'                 => $order->get_order_number(),
			'order_date'               => $order_date,
			'order_status'             => $order->get_status(),
			'order_amount'             => $order->get_total(),
			'order_weight'             => $order_weight,
			'order_shipping_method_id' => $order_shipping_method_id,
			'customer_note'            => $customer_note,
			'customer_email'           => ( WC_DPDFrance_Tools::is_from_wc3() ? $order->get_billing_email() : $order->billing_email ),
			'customer_phone'           => ( WC_DPDFrance_Tools::is_from_wc3() ? $order->get_billing_phone() : $order->billing_phone ),
			'shipping_first_name'      => ( WC_DPDFrance_Tools::is_from_wc3() ? $order->get_shipping_first_name() : $order->shipping_first_name ),
			'shipping_last_name'       => ( WC_DPDFrance_Tools::is_from_wc3() ? $order->get_shipping_last_name() : $order->shipping_last_name ),
			'shipping_company'         => ( WC_DPDFrance_Tools::is_from_wc3() ? $order->get_shipping_company() : $order->shipping_company ),
			'shipping_address_1'       => ( WC_DPDFrance_Tools::is_from_wc3() ? $order->get_shipping_address_1() : $order->shipping_address_1 ),
			'shipping_address_2'       => ( WC_DPDFrance_Tools::is_from_wc3() ? $order->get_shipping_address_2() : $order->shipping_address_2 ),
			'shipping_postcode'        => ( WC_DPDFrance_Tools::is_from_wc3() ? $order->get_shipping_postcode() : $order->shipping_postcode ),
			'shipping_city'            => ( WC_DPDFrance_Tools::is_from_wc3() ? $order->get_shipping_city() : $order->shipping_city ),
			'shipping_country'         => ( WC_DPDFrance_Tools::is_from_wc3() ? $order->get_shipping_country() : $order->shipping_country ),
		];
	}

	/**
	 * * Add global CSS to admin page
	 */
	public function woocommerce_dpdfrance_admin_style() {
		wp_enqueue_style( 'dpdfrance_admin_style', OS_DPDFRANCE_ROOT_URL . '/assets/css/admin/AdminDPDFrance.css');
	}

}

/**
 *  * Add global JS script to checkout page
 */
function woocommerce_dpdfrance_custom_script() {
	wp_enqueue_script( 'dpdfrance_custom_script', OS_DPDFRANCE_ROOT_URL . '/assets/js/front/dpdfrance_main.js', [ 'jquery' ] );
}

/* * Add javascript file to checkout page */
add_action('woocommerce_checkout_shipping', 'woocommerce_dpdfrance_custom_script');

/* * Instantiate module */
$module = new OS_dpdfrance_shipping_wrapper();

/* * Register plugin status hooks */
register_activation_hook( __FILE__, [ $module, 'install' ] );
register_deactivation_hook( __FILE__, [ $module, 'deactivate' ] );

/* * Exec */
$module->OS_dpdfrance_shipping_main();

/* EOF */
